//
// OverheadCamera.cpp
//
// Custom DLL for a runway view 
//
// ROTW from OverHead Chris Wallace Camera example
//
// This code sample illustrates the following aspects of Fly! II SDK programming:
//  * Creating a custom camera view
//  * Updating a custom camera view in real time
//  * Intercepting keystrokes from the sim
//


#include <string.h>
#include <stdio.h>
#include <math.h>

#define FOR_EXPORT
#include "FlyDll.h"

//#define PI   (float)3.1415926

//
// Every custom DLL object must have its own unique signature.  This signature
//   is a 32-bit integer which is written using the 'xxxx' notation in C to
//   provide better readability.  All DLL object signatures must be able to be
//   specified in this way because typically the signature is specified as part
//   of some text control file.  In the case of this DLL, the unique camera ID
//   must be added to the "camera.txt" file.
//
#define RNWY_CAMERA_ID     'rnwy'


//
// Data structure containing all information related to the run-time operation
//   of the DLL.  For this simple camera DLL, we only need to keep track of
//   the camera altitude above the aircraft.
//
typedef struct {
	float  altOffset;
} SDLLData;

//
// Since the callback routine DLLKeyIntercept is not passed a pointer to our
//   DLL object, a pointer to the allocated data structure must be stored
//   globally.
static SDLLData *pGlobalData = NULL;

//
// Globals
SPosition airportCamera, cameraPos;
SAirport ap;
float apDistance, hDistance;
double zoomCam, varAlt = 0;
int choix =0;
char buf[32], ch[4];

//
// Functions
SFlyObjectRef user;
SPosition userPos;

void GetUserAircraftData()
{
	APIGetUserObject(&user);
	APIGetObjectPosition(&user, &userPos);
	APIGetNearestAirport(&userPos, &ap, &apDistance);
}

double pxy, px, py;

double Position2Cam (double tx, double ty)
{
	pxy = atan2(tx, ty);
	return pxy;
}

//
// This function is called by Fly! whenever any of the objects that we have
//   bound in (see DLLInit below) are created.  Since we have added the unique
//   ID of our custom camera object to the cameras.txt control file, and we
//   have bound in the signature value in DLLInit below, when Fly! needs to
//   create the 'cmoh' camera, it will call this function.  It is up to this DLL
//   to actually instantiate the object and manage it in real-time.
//

void DLLInstantiate(const long type, const long id, SDLLObject **object)
{
	if (id == RNWY_CAMERA_ID) {
		if (type == TYPE_DLL_CAMERA) {
			// DLL object type and unique signature match what we are expecting
			*object = APICreateDLLObject();

			// Allocate DLL data structure
			SDLLData *pData = (SDLLData *)APIAllocMem (sizeof(SDLLData));

			// Initialize altitude offset to 1000'
//			pData->altOffset = 1000;

			// Assign a pointer to the DLL data structure into the dllObject
			//   field of the Fly! DLL object.  This allows other callbacks
			//   to access the DLL data.  Also copy the pointer to the static
			//   global for use in DLLKeyIntercept
			(**object).dllObject = (void *)pData;
			pGlobalData = pData;
		}
	}
}

//
// This function is called by Fly! when the camera object we created is being
//   destroyed, i.e. when changing planes or exiting the sim
//
void DLLDestroyObject(SDLLObject *object)
{
	SDLLData *pData = (SDLLData *)(object->dllObject);
	if (!pData) return;

	// DLL data structure must be deallocated to avoid a memory leak
	APIFreeMem (pData);
	pGlobalData = NULL;
}


//
// This function is called by Fly! every time a key is pressed.  It allows
//   the custom DLL to capture keystrokes before they are passed to the
//   sim itself.
//
// We will be capturing the 'Shift +', 'Shift -', 'Ctrl +' and 'Ctrl -' keys
//   only when our custom camera is active.
//

int DLLKeyIntercept (int keyCode, int modifiers)
{
	// Exit immediately if our custom camera has not been instantiated
	if (!pGlobalData) return false;

	// Exit immediately if our custom camera is not active
	if (APIGetCamera() != RNWY_CAMERA_ID) return false;
	
	// Initialize our return code to 'false' indicating that the DLL did not
	//   consume the keypress and that it should be handled by the sim
		bool rc = false;

		switch (modifiers)
		{
			case KB_MODIFIER_CTRL:
				switch (keyCode)
				{
				case KB_KEY_PGUP:
				if(choix==1) varAlt += 0.25;
				rc = true;
				break;
	
				case KB_KEY_PGDN:
				if(choix==1) varAlt -= 0.25;
				rc = true;
				break;

				case KB_KEY_KEYPAD_PLUS:
				zoomCam = 0.999;
				rc = true;
				break;

				case KB_KEY_KEYPAD_MINUS:
				zoomCam = 0.0;
				rc = true;
				break;

				case KB_KEY_LEFT:
				zoomCam -= 0.01;
				if(zoomCam < -2.0) zoomCam = -2.0;
				rc = true;
				break;

				case KB_KEY_RIGHT:
				zoomCam += pow(1-zoomCam, 2) / 10;
				rc = true;
				break;

				case KB_KEY_UP:
				choix = 1;
				rc = true;
				break;

				case KB_KEY_DOWN:
				choix = 0;
				rc = true;
				break;

				case KB_KEY_HOME:
				APIDrawNoticeToUser(ap.name, 10.0);
				rc = true;
				break;
				}
			break;
		}
						if (choix ==1) strcpy(ch,"up");
						else strcpy(ch,"do");
						sprintf(buf, "zoom : %.2f - pos : %s", zoomCam, ch);
						APIDrawNoticeToUser(buf, 1.0);
		return rc;

}

//
// This function is called on every redraw frame when the custom camera
//   is active.
//
// Inputs:
//    *object            Target sim object (typically the user aircraft)
//    *targetPosition    Position of the target object (do not modify!)
//    *targetOrientation Orientation of the target object (do not modify!)
//    deltaTime          Elapsed time since previous redraw frame
// 
// Outputs:
//    *eyePosition       Position of the camera eye in 3D space (lat/lon/alt)
//    *eyeOrientation    Orientation of the camera eye (pitch/bank/heading)
//
void DLLUpdateCamera(SDLLObject *object,
					 SPosition *targetPosition,
					 SVector *targetOrientation,
					 SPosition *eyePosition,
					 SVector *eyeOrientation,
					 const float deltaTime)
{
	SDLLData *pData = (SDLLData *)(object->dllObject);
	if (!pData) return;

	GetUserAircraftData();

	airportCamera = ap.pos;

	px = userPos.lon - airportCamera.lon;
	py = userPos.lat - airportCamera.lat;

	// Camera Position
	cameraPos.lat = airportCamera.lat + (zoomCam *(py));
	cameraPos.lon = airportCamera.lon + (zoomCam *(px));
							// Stop when far than 2100.0
							if(abs((int)px) >=2100 || abs((int)py) >= 2100)
							{
								eyePosition->lat = airportCamera.lat;
								eyePosition->lon = airportCamera.lon;
								eyePosition->alt = airportCamera.alt + 7500;
								APIPanCamera((float)1.57, (float)pxy, 0);
								APIDrawNoticeToUser(ap.name, 10.0);
								return;
							}
	if(choix == 1) cameraPos.alt = userPos.alt + varAlt + 5.0;
	else cameraPos.alt = (25 * pow(zoomCam - 1, 2)) + (zoomCam * userPos.alt);

	// Eye Position
	eyePosition->lat = cameraPos.lat;
	eyePosition->lon = cameraPos.lon;
	eyePosition->alt = cameraPos.alt;
	
	// Eye Orientation
	Position2Cam(px, py);
	hDistance = APIGreatCircleDistance(&cameraPos, &userPos);
	APIPanCamera(float(atan((cameraPos.alt - userPos.alt) / hDistance)), (float)pxy, 0);
	eyeOrientation->z = 0.0;

	// Ensure eye position never goes underground
	float groundAlt = APIGetGroundHeight (eyePosition);
	if(eyePosition->alt < groundAlt) eyePosition->alt = groundAlt + 5.0;
}

//
//  This function is called by Fly! to determine whether the camera is an internal
//    or external camera, and therefore whether to display the interior panels,
//    or the external model.  Since our custom camera is always external, this
//    function simply returns a hardcoded value of zero.
//
int DLLIsInteriorCamera (SDLLObject *object)
{
	return 0;
}

void DLLGetCameraName(SDLLObject *object, char *name, int maxLength)
{
	strncpy(name, "Runway View", maxLength);
}

void DLLStartSituation(void)
{
	memset(&ap, 0, sizeof(SAirport));
}

/////////////////////////////////////////////////////////////
//
//	DLL Setup (Init & Kill)
//
//	These functions MUST be implemented, or the DLL
//	will totally fail!!!!
//
/////////////////////////////////////////////////////////////

int	DLLInit(DLLFunctionRegistry *dll,
			SDLLCopyright *copyright,
			SDLLRegisterTypeList **types)
{
	//
	//	populate copyright info
	//
	strcpy(copyright->product, "Runway Camera DLL");
	strcpy(copyright->company, "ROTW");
	strcpy(copyright->programmer, "Laurent Claudet from Chris Wallace example");
	strcpy(copyright->dateTimeVersion, "1.00 28/04/02");
	strcpy(copyright->email, "laurent.claudet@bad.email");
	copyright->internalVersion = 100;

	//
	//	return function pointers of available DLL functions
	//
	dll->DLLInstantiate = DLLInstantiate;
	dll->DLLDestroyObject = DLLDestroyObject;
	dll->DLLKeyIntercept = DLLKeyIntercept;
	dll->DLLUpdateCamera = DLLUpdateCamera;
	dll->DLLIsInteriorCamera = DLLIsInteriorCamera;
	dll->DLLGetCameraName = DLLGetCameraName;
	dll->DLLStartSituation = DLLStartSituation;

	//	Register all supported object types
	APIAddRegisteredType (types, TYPE_DLL_CAMERA, RNWY_CAMERA_ID);

	return(1);
}

void DLLKill(void)
{
}

